
from dataclasses import dataclass
from typing import Tuple, Optional
import numpy as np

@dataclass
class SceneConfig:
    W: int
    H: int
    roi_bbox: Tuple[int,int,int,int]  # x0,y0,x1,y1
    w_inner: int
    allowed_mask: Optional[np.ndarray] = None  # HxW boolean; None => fully allowed

def make_optics_scene(L_out: int, w: int) -> SceneConfig:
    W = int(1.5*L_out)
    H = int(0.75*L_out)
    roi_h = max(1, int(0.25*L_out))
    cx, cy = W//2, H//2
    x0 = cx - L_out//2
    x1 = x0 + L_out
    y0 = cy - roi_h//2
    y1 = y0 + roi_h
    return SceneConfig(W=W, H=H, roi_bbox=(x0,y0,x1,y1), w_inner=w)

def make_optics_roi(scene: SceneConfig):
    return np.zeros((scene.H, scene.W), dtype=int)

def place_sources_from_s(scene: SceneConfig, s: int, y_row: int):
    cx = scene.W//2
    xL = max(0, min(scene.W-1, int(round(cx - s/2))))
    xR = max(0, min(scene.W-1, int(round(cx + s/2))))
    return (xL, y_row), (xR, y_row)

def profile_from_roi(scene: SceneConfig, screen: np.ndarray):
    x0,y0,x1,y1 = scene.roi_bbox
    roi = screen[y0:y1, x0:x1]
    return roi.sum(axis=0).astype(int)

def find_peaks_trough(profile):
    if profile.size < 3: return 0,0,0
    mid = profile.size//2
    p1 = int(profile[:mid].max()) if mid>0 else 0
    p2 = int(profile[mid:].max()) if mid<profile.size else 0
    i1 = int(np.argmax(profile[:mid])) if mid>0 else 0
    i2 = mid + int(np.argmax(profile[mid:])) if mid<profile.size else mid
    lo = min(i1, i2); hi = max(i1, i2)
    if hi - lo <= 1:
        t = int(min(profile[lo], profile[hi]) if hi<profile.size else profile[lo])
    else:
        t = int(profile[lo:hi+1].min())
    return p1, p2, t

def make_two_rail_mask(scene: SceneConfig, xL: int, xR: int, halfwidth: int = 0):
    mask = np.zeros((scene.H, scene.W), dtype=bool)
    for x0 in (xL, xR):
        x_lo = max(0, x0 - halfwidth)
        x_hi = min(scene.W - 1, x0 + halfwidth)
        mask[:, x_lo:x_hi+1] = True
    return mask

def make_central_corridor_mask(scene: SceneConfig, halfwidth: int = 0):
    # allow only a narrow vertical corridor around ROI center x
    x0,y0,x1,y1 = scene.roi_bbox
    cx = (x0 + x1)//2
    mask = np.zeros((scene.H, scene.W), dtype=bool)
    lo = max(0, cx - halfwidth)
    hi = min(scene.W-1, cx + halfwidth)
    mask[:, lo:hi+1] = True
    return mask
